import React, { useState, useEffect } from 'react';
import { useSearchParams, useNavigate } from 'react-router-dom';
import DashboardLayout from '../layouts/DashboardLayout';
import { useAuth, api } from '../App';
import { Button } from '../components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '../components/ui/card';
import { Input } from '../components/ui/input';
import { Label } from '../components/ui/label';
import { Badge } from '../components/ui/badge';
import { toast } from 'sonner';
import { Check, Crown, CreditCard, User } from 'lucide-react';

const PLANS = {
  free: { name: 'Free', price: 0, links: 3 },
  basic: { name: 'Basic', price: 9.99, links: 30 },
  pro: { name: 'Pro', price: 29.99, links: 300 },
  ultimate: { name: 'Ultimate', price: 79.99, links: 1000 },
};

const SettingsPage = () => {
  const { user, refreshUser } = useAuth();
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const [loading, setLoading] = useState(false);
  const [name, setName] = useState(user?.name || '');

  useEffect(() => {
    // Check for payment success
    const sessionId = searchParams.get('session_id');
    const status = searchParams.get('status');
    
    if (sessionId && status === 'success') {
      pollPaymentStatus(sessionId);
    }
  }, [searchParams]);

  const pollPaymentStatus = async (sessionId, attempts = 0) => {
    if (attempts >= 5) {
      toast.error('Payment verification timed out. Please contact support.');
      return;
    }

    try {
      const res = await api.get(`/checkout/status/${sessionId}`);
      if (res.data.payment_status === 'paid') {
        toast.success('Subscription upgraded successfully!');
        await refreshUser();
        navigate('/settings', { replace: true });
      } else {
        setTimeout(() => pollPaymentStatus(sessionId, attempts + 1), 2000);
      }
    } catch (error) {
      console.error('Payment check failed:', error);
    }
  };

  const handleUpgrade = async (planId) => {
    setLoading(true);
    try {
      const res = await api.post('/checkout/subscribe', {
        plan_id: planId,
        origin_url: window.location.origin,
        payment_method: 'card',
      });
      window.location.href = res.data.url;
    } catch (error) {
      toast.error(error.response?.data?.detail || 'Failed to start checkout');
      setLoading(false);
    }
  };

  const handleUpgradeWithCrypto = async (planId) => {
    setLoading(true);
    try {
      const res = await api.post('/checkout/subscribe', {
        plan_id: planId,
        origin_url: window.location.origin,
        payment_method: 'crypto',
      });
      window.location.href = res.data.url;
    } catch (error) {
      toast.error(error.response?.data?.detail || 'Failed to start checkout');
      setLoading(false);
    }
  };

  const currentPlan = PLANS[user?.subscription_plan] || PLANS.free;

  return (
    <DashboardLayout title="Settings">
      <div className="space-y-6 fade-in max-w-4xl">
        {/* Profile Section */}
        <Card className="border-2">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <User className="h-5 w-5" />
              Profile
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="space-y-2">
              <Label>Email</Label>
              <Input value={user?.email} disabled />
            </div>
            <div className="space-y-2">
              <Label>Name</Label>
              <Input
                value={name}
                onChange={(e) => setName(e.target.value)}
                data-testid="profile-name-input"
              />
            </div>
            <div className="flex items-center gap-2">
              <Badge variant={user?.role === 'admin' ? 'default' : 'secondary'}>
                {user?.role === 'admin' ? 'Admin' : 'User'}
              </Badge>
              <Badge variant="outline">
                Member since {new Date(user?.created_at).toLocaleDateString()}
              </Badge>
            </div>
          </CardContent>
        </Card>

        {/* Current Plan */}
        <Card className="border-2">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Crown className="h-5 w-5 text-primary" />
              Current Plan
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex items-center justify-between">
              <div>
                <h3 className="text-2xl font-bold">{currentPlan.name}</h3>
                <p className="text-muted-foreground">
                  {user?.links_created_this_month || 0} / {currentPlan.links === -1 ? '∞' : currentPlan.links} links this month
                </p>
              </div>
              <div className="text-right">
                <p className="text-3xl font-bold">${currentPlan.price}</p>
                <p className="text-muted-foreground">/month</p>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Upgrade Options */}
        {user?.subscription_plan !== 'ultimate' && user?.subscription_plan !== 'enterprise' && (
          <Card className="border-2">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <CreditCard className="h-5 w-5" />
                Upgrade Plan
              </CardTitle>
              <CardDescription>
                Get more links and premium features
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="grid md:grid-cols-3 gap-4">
                {Object.entries(PLANS)
                  .filter(([key]) => {
                    const planOrder = ['free', 'basic', 'pro', 'ultimate'];
                    const currentIndex = planOrder.indexOf(user?.subscription_plan || 'free');
                    const planIndex = planOrder.indexOf(key);
                    return planIndex > currentIndex && key !== 'enterprise';
                  })
                  .map(([key, plan]) => (
                    <Card
                      key={key}
                      className={`border-2 ${key === 'pro' ? 'border-primary' : ''}`}
                    >
                      <CardHeader className="pb-2">
                        <CardTitle className="text-lg">{plan.name}</CardTitle>
                        <CardDescription>
                          <span className="text-2xl font-bold text-foreground">${plan.price}</span>
                          /month
                        </CardDescription>
                      </CardHeader>
                      <CardContent className="space-y-3">
                        <p className="text-sm">{plan.links} links/month</p>
                        <div className="space-y-2">
                          <Button
                            className="w-full"
                            onClick={() => handleUpgrade(key)}
                            disabled={loading}
                            data-testid={`upgrade-${key}-btn`}
                          >
                            Pay with Card
                          </Button>
                          <Button
                            className="w-full"
                            variant="outline"
                            onClick={() => handleUpgradeWithCrypto(key)}
                            disabled={loading}
                            data-testid={`upgrade-${key}-crypto-btn`}
                          >
                            Pay with Crypto
                          </Button>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
              </div>
            </CardContent>
          </Card>
        )}

        {/* Billing Info */}
        <Card className="border-2">
          <CardHeader>
            <CardTitle>Billing</CardTitle>
          </CardHeader>
          <CardContent>
            <p className="text-muted-foreground">
              Manage your billing information and view payment history.
            </p>
            <p className="text-sm text-muted-foreground mt-4">
              For enterprise plans and custom billing, please contact our sales team.
            </p>
          </CardContent>
        </Card>
      </div>
    </DashboardLayout>
  );
};

export default SettingsPage;
