import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { useTheme } from '../App';
import { Button } from '../components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '../components/ui/card';
import { Input } from '../components/ui/input';
import { toast } from 'sonner';
import { Toaster } from '../components/ui/sonner';
import axios from 'axios';
import { Link2, Lock, Moon, Sun, ExternalLink, AlertCircle, Clock } from 'lucide-react';

const API_URL = process.env.REACT_APP_BACKEND_URL;

const RedirectPage = () => {
  const { shortCode } = useParams();
  const navigate = useNavigate();
  const { theme, toggleTheme } = useTheme();
  const [loading, setLoading] = useState(true);
  const [linkInfo, setLinkInfo] = useState(null);
  const [error, setError] = useState(null);
  const [password, setPassword] = useState('');
  const [verifying, setVerifying] = useState(false);
  const [countdown, setCountdown] = useState(5);
  const [showAd, setShowAd] = useState(true);

  useEffect(() => {
    fetchLinkInfo();
  }, [shortCode]);

  useEffect(() => {
    if (linkInfo && !linkInfo.requires_password && showAd && countdown > 0) {
      const timer = setTimeout(() => setCountdown(countdown - 1), 1000);
      return () => clearTimeout(timer);
    }
    
    if (linkInfo && !linkInfo.requires_password && countdown === 0) {
      handleRedirect();
    }
  }, [countdown, linkInfo, showAd]);

  const fetchLinkInfo = async () => {
    try {
      const res = await axios.get(`${API_URL}/api/public/link/${shortCode}`);
      setLinkInfo(res.data);
    } catch (error) {
      if (error.response?.status === 404) {
        setError('Link not found');
      } else if (error.response?.status === 410) {
        setError('This link has expired');
      } else {
        setError('Failed to load link');
      }
    } finally {
      setLoading(false);
    }
  };

  const handlePasswordSubmit = async (e) => {
    e.preventDefault();
    if (!password) return;

    setVerifying(true);
    try {
      const res = await axios.post(`${API_URL}/s/${shortCode}/verify`, { password });
      if (res.data.valid) {
        window.location.href = res.data.target_url;
      }
    } catch (error) {
      toast.error('Invalid password');
    } finally {
      setVerifying(false);
    }
  };

  const handleRedirect = () => {
    window.location.href = `${API_URL}/s/${shortCode}`;
  };

  const handleSkipAd = () => {
    setShowAd(false);
    handleRedirect();
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-background">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="min-h-screen flex flex-col bg-background">
        <header className="p-4 flex justify-between items-center">
          <div className="flex items-center gap-2">
            <Link2 className="h-6 w-6 text-primary" />
            <span className="font-bold text-xl">Boltlytics</span>
          </div>
          <Button variant="ghost" size="icon" onClick={toggleTheme}>
            {theme === 'dark' ? <Sun className="h-5 w-5" /> : <Moon className="h-5 w-5" />}
          </Button>
        </header>
        <main className="flex-1 flex items-center justify-center p-4">
          <Card className="w-full max-w-md border-2">
            <CardContent className="pt-6 text-center">
              <AlertCircle className="h-16 w-16 mx-auto text-destructive mb-4" />
              <h2 className="text-2xl font-bold mb-2">Oops!</h2>
              <p className="text-muted-foreground mb-6">{error}</p>
              <Button onClick={() => navigate('/')}>Go Home</Button>
            </CardContent>
          </Card>
        </main>
        <Toaster position="top-right" />
      </div>
    );
  }

  // Password required
  if (linkInfo?.requires_password) {
    return (
      <div className="min-h-screen flex flex-col bg-background">
        <header className="p-4 flex justify-between items-center">
          <div className="flex items-center gap-2">
            <Link2 className="h-6 w-6 text-primary" />
            <span className="font-bold text-xl">Boltlytics</span>
          </div>
          <Button variant="ghost" size="icon" onClick={toggleTheme}>
            {theme === 'dark' ? <Sun className="h-5 w-5" /> : <Moon className="h-5 w-5" />}
          </Button>
        </header>
        <main className="flex-1 flex items-center justify-center p-4">
          <Card className="w-full max-w-md border-2">
            <CardHeader className="text-center">
              <div className="mx-auto w-16 h-16 bg-primary/10 rounded-full flex items-center justify-center mb-4">
                <Lock className="h-8 w-8 text-primary" />
              </div>
              <CardTitle>Password Protected</CardTitle>
              <CardDescription>
                This link requires a password to access
              </CardDescription>
            </CardHeader>
            <CardContent>
              <form onSubmit={handlePasswordSubmit} className="space-y-4">
                <Input
                  type="password"
                  placeholder="Enter password"
                  value={password}
                  onChange={(e) => setPassword(e.target.value)}
                  data-testid="link-password-input"
                />
                <Button
                  type="submit"
                  className="w-full"
                  disabled={verifying}
                  data-testid="verify-password-btn"
                >
                  {verifying ? 'Verifying...' : 'Continue'}
                </Button>
              </form>
            </CardContent>
          </Card>
        </main>
        <Toaster position="top-right" />
      </div>
    );
  }

  // Interstitial Ad Page
  return (
    <div className="min-h-screen flex flex-col bg-background">
      <header className="p-4 flex justify-between items-center">
        <div className="flex items-center gap-2">
          <Link2 className="h-6 w-6 text-primary" />
          <span className="font-bold text-xl">Boltlytics</span>
        </div>
        <Button variant="ghost" size="icon" onClick={toggleTheme}>
          {theme === 'dark' ? <Sun className="h-5 w-5" /> : <Moon className="h-5 w-5" />}
        </Button>
      </header>
      
      <main className="flex-1 flex items-center justify-center p-4">
        <Card className="w-full max-w-xl border-2">
          <CardHeader className="text-center">
            <CardTitle className="text-2xl">Redirecting you...</CardTitle>
            <CardDescription>
              {linkInfo?.title || 'Your destination is ready'}
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-6">
            {/* Ad Placeholder */}
            <div className="bg-muted rounded-lg p-8 text-center">
              <p className="text-muted-foreground text-sm mb-2">Advertisement</p>
              <div className="bg-gradient-to-r from-primary/20 to-primary/10 rounded-lg p-6">
                <h3 className="text-lg font-bold mb-2">Start Shortening Links Today!</h3>
                <p className="text-sm text-muted-foreground mb-4">
                  Create your own short links with advanced analytics
                </p>
                <Button variant="outline" size="sm" onClick={() => navigate('/register')}>
                  Try Boltlytics Free
                </Button>
              </div>
            </div>

            {/* Destination Preview */}
            <div className="p-4 bg-muted rounded-lg">
              <p className="text-sm text-muted-foreground mb-1">You will be redirected to:</p>
              <p className="text-sm font-mono truncate">{linkInfo?.target_url}</p>
            </div>

            {/* Countdown & Actions */}
            <div className="flex flex-col items-center gap-4">
              <div className="flex items-center gap-2 text-lg">
                <Clock className="h-5 w-5 text-primary" />
                <span>Redirecting in <strong>{countdown}</strong> seconds</span>
              </div>
              
              <div className="flex gap-3">
                <Button 
                  onClick={handleSkipAd}
                  data-testid="skip-ad-btn"
                >
                  <ExternalLink className="mr-2 h-4 w-4" />
                  Skip & Continue
                </Button>
                <Button 
                  variant="outline" 
                  onClick={() => navigate('/')}
                  data-testid="go-home-btn"
                >
                  Cancel
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>
      </main>

      {/* Powered by footer */}
      <footer className="p-4 text-center">
        <p className="text-sm text-muted-foreground">
          Powered by <span className="text-primary font-medium">Boltlytics</span> - 
          The professional URL shortener
        </p>
      </footer>
      <Toaster position="top-right" />
    </div>
  );
};

export default RedirectPage;
