import React from 'react';
import { Link, useNavigate } from 'react-router-dom';
import { useTheme, useAuth } from '../App';
import { Button } from '../components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '../components/ui/card';
import { Link2, Moon, Sun, Check, ArrowRight, Shield, Zap } from 'lucide-react';

const PLANS = {
  free: { 
    name: 'Free', 
    price: 0, 
    links: 3, 
    features: ['3 links per month', 'Basic analytics', 'Standard support', 'QR codes'] 
  },
  basic: { 
    name: 'Basic', 
    price: 9.99, 
    links: 30, 
    features: ['30 links per month', 'Advanced analytics', 'Priority support', 'QR codes', 'Link history'] 
  },
  pro: { 
    name: 'Pro', 
    price: 29.99, 
    links: 300, 
    features: ['300 links per month', 'Full analytics', 'API access', 'Password protection', 'Custom aliases', 'Priority support'] 
  },
  ultimate: { 
    name: 'Ultimate', 
    price: 79.99, 
    links: 1000, 
    features: ['1000 links per month', 'Everything in Pro', 'Custom domains', 'White-label branding', 'Dedicated support', 'Advanced security'] 
  },
};

const PricingPage = () => {
  const { theme, toggleTheme } = useTheme();
  const { user } = useAuth();
  const navigate = useNavigate();

  const handleSelectPlan = (planId) => {
    if (user) {
      navigate('/settings');
    } else {
      navigate('/register');
    }
  };

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <header className="sticky top-0 z-50 glass border-b">
        <div className="max-w-7xl mx-auto px-4 md:px-8">
          <div className="flex h-16 items-center justify-between">
            <Link to="/" className="flex items-center gap-2" data-testid="logo">
              <Link2 className="h-6 w-6 text-primary" />
              <span className="font-bold text-xl tracking-tight">Boltlytics</span>
            </Link>

            <nav className="hidden md:flex items-center gap-8">
              <Link to="/#features" className="text-muted-foreground hover:text-foreground transition-colors">
                Features
              </Link>
              <Link to="/pricing" className="text-foreground font-medium">
                Pricing
              </Link>
              {user ? (
                <Link to="/dashboard" className="text-muted-foreground hover:text-foreground transition-colors">
                  Dashboard
                </Link>
              ) : (
                <Link to="/login" className="text-muted-foreground hover:text-foreground transition-colors">
                  Login
                </Link>
              )}
            </nav>

            <div className="flex items-center gap-3">
              <Button
                variant="ghost"
                size="icon"
                onClick={toggleTheme}
                data-testid="theme-toggle"
              >
                {theme === 'dark' ? <Sun className="h-5 w-5" /> : <Moon className="h-5 w-5" />}
              </Button>
              {!user && (
                <Button onClick={() => navigate('/register')} data-testid="get-started-btn">
                  Get Started
                </Button>
              )}
            </div>
          </div>
        </div>
      </header>

      {/* Main Content */}
      <main className="py-20 px-4 md:px-8">
        <div className="max-w-7xl mx-auto">
          {/* Header */}
          <div className="text-center mb-16">
            <h1 className="text-4xl md:text-6xl font-black tracking-tighter mb-4">
              Simple, Transparent Pricing
            </h1>
            <p className="text-lg md:text-xl text-muted-foreground max-w-2xl mx-auto">
              Choose the plan that fits your needs. All plans include core features.
              Upgrade or downgrade anytime.
            </p>
          </div>

          {/* Plans Grid */}
          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6 mb-16">
            {Object.entries(PLANS).map(([key, plan]) => (
              <Card 
                key={key} 
                className={`hover-lift border-2 ${key === 'pro' ? 'border-primary shadow-lg shadow-primary/20' : ''}`}
              >
                <CardHeader>
                  {key === 'pro' && (
                    <div className="flex items-center gap-1 text-xs font-medium text-primary mb-2">
                      <Zap className="h-3 w-3" />
                      MOST POPULAR
                    </div>
                  )}
                  <CardTitle className="text-2xl">{plan.name}</CardTitle>
                  <CardDescription>
                    <span className="text-4xl font-bold text-foreground">${plan.price}</span>
                    <span className="text-muted-foreground">/month</span>
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <ul className="space-y-3">
                    {plan.features.map((feature, i) => (
                      <li key={i} className="flex items-start gap-2 text-sm">
                        <Check className="h-4 w-4 text-primary mt-0.5 shrink-0" />
                        <span>{feature}</span>
                      </li>
                    ))}
                  </ul>
                  <Button 
                    className="w-full" 
                    variant={key === 'pro' ? 'default' : 'outline'}
                    onClick={() => handleSelectPlan(key)}
                    data-testid={`select-${key}-btn`}
                  >
                    {plan.price === 0 ? 'Get Started Free' : 'Subscribe'}
                    <ArrowRight className="ml-2 h-4 w-4" />
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>

          {/* Enterprise CTA */}
          <Card className="border-2 max-w-3xl mx-auto">
            <CardContent className="flex flex-col md:flex-row items-center gap-6 p-8">
              <div className="h-16 w-16 bg-primary/10 rounded-full flex items-center justify-center shrink-0">
                <Shield className="h-8 w-8 text-primary" />
              </div>
              <div className="flex-1 text-center md:text-left">
                <h3 className="text-2xl font-bold mb-2">Enterprise</h3>
                <p className="text-muted-foreground">
                  Need unlimited links, custom integrations, SLA, and dedicated support?
                  Let's talk about a custom solution for your organization.
                </p>
              </div>
              <Button size="lg" variant="outline" data-testid="contact-sales-btn">
                Contact Sales
              </Button>
            </CardContent>
          </Card>

          {/* FAQ Section */}
          <div className="mt-20 max-w-3xl mx-auto">
            <h2 className="text-3xl font-bold text-center mb-10">
              Frequently Asked Questions
            </h2>
            <div className="space-y-6">
              {[
                {
                  q: 'Can I switch plans anytime?',
                  a: 'Yes! You can upgrade or downgrade your plan at any time. Changes take effect immediately.',
                },
                {
                  q: 'What payment methods do you accept?',
                  a: 'We accept all major credit cards and cryptocurrency payments through Stripe.',
                },
                {
                  q: 'Is there a free trial?',
                  a: 'Our Free plan is free forever! You can use it to test all core features before upgrading.',
                },
                {
                  q: 'What happens if I exceed my link limit?',
                  a: "You'll need to upgrade to a higher plan or wait until the next billing cycle to create more links.",
                },
              ].map((faq, i) => (
                <Card key={i} className="border-2">
                  <CardHeader>
                    <CardTitle className="text-lg">{faq.q}</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <p className="text-muted-foreground">{faq.a}</p>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </div>
      </main>

      {/* Footer */}
      <footer className="border-t py-12">
        <div className="max-w-7xl mx-auto px-4 md:px-8">
          <div className="flex flex-col md:flex-row justify-between items-center gap-4">
            <div className="flex items-center gap-2">
              <Link2 className="h-5 w-5 text-primary" />
              <span className="font-bold">Boltlytics</span>
            </div>
            <p className="text-sm text-muted-foreground">
              © 2024 Boltlytics. All rights reserved.
            </p>
          </div>
        </div>
      </footer>
    </div>
  );
};

export default PricingPage;
